<?php
/**
 * CharacterDAO - Handle character operations
 */

class CharacterDAO {
    private $db;
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    /**
     * Get characters list
     * CharacterDAO.getCharactersList(session_key)
     */
    public function getCharactersList($sessionKey) {
        $account = validateSessionKey($sessionKey);
        if (!$account) {
            throw new Exception("Invalid session");
        }
        
        $stmt = $this->db->prepare("
            SELECT character_id, character_name, character_level, character_xp,
                   character_hp, character_max_hp, character_cp, character_max_cp,
                   character_gold, character_token, character_rank, character_gender,
                   character_face, character_hair, character_body_set,
                   character_equipped_weapon, character_equipped_back_item,
                   character_wind, character_fire, character_lightning,
                   character_water, character_earth, character_tp,
                   character_class,
                   character_skill_talent
            FROM characters 
            WHERE account_id = ?
        ");
        $stmt->execute([$account['account_id']]);
        return $stmt->fetchAll();
    }
    
    /**
     * Get character by ID
     * CharacterDAO.getCharacterById(session_key, character_id)
     */
    public function getCharacterById($sessionKey, $characterId) {
        // Khusus panggilan dari PvP (type "recruit"), client mengirim string
        // "recruit" sebagai sessionKey. Dalam kasus ini kita tidak bisa
        // memvalidasi sessionKey terhadap account_id, karena host sedang
        // meminta data karakter musuh. Untuk itu, bila sessionKey ===
        // 'recruit' kita lewati validateSessionKey dan ambil karakter
        // hanya berdasarkan character_id.

        $account = null;

        if ($sessionKey !== 'recruit') {
            $account = validateSessionKey($sessionKey);
            if (!$account) {
                throw new Exception("Invalid session");
            }
        }

        if ($sessionKey === 'recruit') {
            $stmt = $this->db->prepare("
                SELECT * FROM characters 
                WHERE character_id = ?
            ");
            $stmt->execute([$characterId]);
        } else {
            $stmt = $this->db->prepare("
                SELECT * FROM characters 
                WHERE character_id = ? AND account_id = ?
            ");
            $stmt->execute([$characterId, $account['account_id']]);
        }

        $character = $stmt->fetch();
        
        if (!$character) {
            throw new Exception("Character not found");
        }
        
        // Parse JSON fields
        // Parse JSON fields (PHP 5.5 compatible)
$character['character_skills'] = json_decode(
    isset($character['character_skills']) ? $character['character_skills'] : '[]',
    true
);

$character['character_skill_talent'] = json_decode(
    isset($character['character_skill_talent']) ? $character['character_skill_talent'] : '[]',
    true
);

$character['character_skill_resistance'] = json_decode(
    isset($character['character_skill_resistance']) ? $character['character_skill_resistance'] : '[]',
    true
);

$character['character_inventory'] = json_decode(
    isset($character['character_inventory']) ? $character['character_inventory'] : '{}',
    true
);

$character['character_body_parts'] = json_decode(
    isset($character['character_body_parts']) ? $character['character_body_parts'] : '{}',
    true
);

$character['character_senjutsu'] = json_decode(
    isset($character['character_senjutsu']) ? $character['character_senjutsu'] : '[]',
    true
);

        // $character['character_skills'] = json_decode($character['character_skills'] ?? '[]', true);
        // $character['character_skill_talent'] = json_decode($character['character_skill_talent'] ?? '[]', true);
        // $character['character_skill_resistance'] = json_decode($character['character_skill_resistance'] ?? '[]', true);
        // $character['character_inventory'] = json_decode($character['character_inventory'] ?? '{}', true);
        // $character['character_body_parts'] = json_decode($character['character_body_parts'] ?? '{}', true);
        // $character['character_senjutsu'] = json_decode($character['character_senjutsu'] ?? '[]', true);
                // ------- Build character_inv_slots untuk client -------
        // Default slot kalau tidak ada data
        $defaultSlots = array(
            'weapon'   => 1,
            'armor'    => 1,
            'back'     => 1,
            'item'     => 0,
            'essence'  => 0,
            'material' => 0,
            'pet'      => 0,
        );

        if (!empty($character['inv_slots_obj'])) {
            $slots = json_decode($character['inv_slots_obj'], true);
            if (is_array($slots)) {
                $character['character_inv_slots'] = array_merge($defaultSlots, $slots);
            } else {
                $character['character_inv_slots'] = $defaultSlots;
            }
        } else {
            $character['character_inv_slots'] = $defaultSlots;
        }

               // ------- Build expiry_data agar parseRawCharacter tidak Error #1010 -------
        $expiryData = new stdClass();

        // Untuk sekarang, semua array kosong (tidak ada item expiry)
        $expiryData->remove_inv_arr     = array();
        $expiryData->add_inv_arr        = array();
        $expiryData->equip_arr          = array();
        $expiryData->remove_equip_arr   = array();
        $expiryData->current_expiry_arr = array();
        $expiryData->expiry_pet_data    = array();

        // Bangun hashStr persis seperti di DataParser.as
        $hashStr =
            implode(',', $expiryData->remove_inv_arr) .
            implode(',', $expiryData->add_inv_arr) .
            implode(',', $expiryData->equip_arr) .
            implode(',', $expiryData->current_expiry_arr) .
            implode(',', $expiryData->remove_equip_arr);

        // Hitung expiry_hash dengan algoritma yang sama seperti ClientLibrary.getHash
        $sessionKeyStr = (string)$sessionKey; // argumen pertama getCharacterById

        $secret = 'Vmn34aAciYK00Hen26nT01';
        $base   = $hashStr . $secret . $sessionKeyStr;
        $sha1   = sha1($base);

        $offset = 0;
        if (strlen($sessionKeyStr) > 1) {
            $hexChar = substr($sessionKeyStr, 1, 1);
            if (ctype_xdigit($hexChar)) {
                $offset = hexdec($hexChar);
            }
        }

        $expiryData->expiry_hash = substr($sha1, $offset, 12);

   

                // ------- Normalisasi field karakter yang dipakai untuk hash -------
        $defaults = array(
            'character_id'               => 0,
            'character_level'            => 1,
            'character_xp'               => 0,
            'character_rank'             => 0,
            'character_gold'             => 0,
            'character_armor'            => 0,
            'character_equipped_skills'  => '',
            'character_equipped_weapon'  => '',
            'character_equipped_body_set'=> '',
            'character_body_set'         => '',
            'character_item'             => '',
            'character_weapon'           => '',
            'character_skill'            => '',
            'character_magatama'         => '',
            'character_npc'              => '',
            'character_equipped_back_item' => '',
            'character_back_item'        => '',
            'character_equipped_accessory'=> '',
            'character_accessory'        => '',
            'character_hair_color'       => 0,
            'character_gender'           => 0,
            'character_skin_color'       => 0,
            'character_face'             => '',
            'character_hair'             => '',
            'character_fire'             => 0,
            'character_water'            => 0,
            'character_wind'             => 0,
            'character_earth'            => 0,
            'character_lightning'        => 0,
            'character_taijutsu'         => 0,
            'character_genjutsu'         => 0,
            'character_summon'           => 0,
            'character_control'          => 0,
            'character_bloodline'        => '',
            'character_mission'          => '',
            'character_ninja_essence'    => '',
            'character_material'         => '',
            'character_inv_hair'         => '',
        );

        foreach ($defaults as $key => $def) {
            if (!isset($character[$key]) || $character[$key] === null) {
                $character[$key] = $def;
            }
        }

        // Khusus hair color: kalau disimpan sebagai array [0,0], ubah ke string "0,0"
        if (isset($character['character_hair_color']) && is_array($character['character_hair_color'])) {
            $character['character_hair_color'] = implode(',', $character['character_hair_color']);
        }

                // ------- Hitung character_pre_hash -------
        $characterPreStr =
            (string)$character['character_level'] . ',' .
            (string)$character['character_xp'] . ',' .
            (string)$character['character_rank'] . ',' .
            (string)$character['character_equipped_skills'] . ',' .
            (string)$character['character_fire'] . ',' .
            (string)$character['character_water'] . ',' .
            (string)$character['character_earth'] . ',' .
            (string)$character['character_lightning'];

        // Algoritma sama seperti ClientLibrary.getHash(sessionKey, payload)
        $sessionKeyStr = (string)$sessionKey;
        $secret        = 'Vmn34aAciYK00Hen26nT01';
        $basePre       = $characterPreStr . $secret . $sessionKeyStr;
        $sha1Pre       = sha1($basePre);

        $offsetPre = 0;
        if (strlen($sessionKeyStr) > 1) {
            $hexCharPre = substr($sessionKeyStr, 1, 1);
            if (ctype_xdigit($hexCharPre)) {
                $offsetPre = hexdec($hexCharPre);
            }
        }

        $character['character_pre_hash'] = substr($sha1Pre, $offsetPre, 12);

                // ------- Hitung character_hash -------
        $characterStr =
            (string)$character['character_id'] .
            (string)$character['character_level'] .
            (string)$character['character_xp'] .
            (string)$character['character_rank'] .
            (string)$character['character_gold'] .
            (string)$character['character_armor'] .
            (string)$character['character_equipped_skills'] .
            (string)$character['character_equipped_weapon'] .
            (string)$character['character_equipped_body_set'] .
            (string)$character['character_body_set'] .
            (string)$character['character_item'] .
            (string)$character['character_weapon'] .
            (string)$character['character_skill'] .
            (string)$character['character_magatama'] .
            (string)$character['character_npc'] .
            (string)$character['character_equipped_back_item'] .
            (string)$character['character_back_item'] .
            (string)$character['character_equipped_accessory'] .
            (string)$character['character_accessory'] .
            (string)$character['character_hair_color'] .
            (string)$character['character_gender'] .
            (string)$character['character_skin_color'] .
            (string)$character['character_face'] .
            (string)$character['character_hair'] .
            (string)$character['character_fire'] .
            (string)$character['character_water'] .
            (string)$character['character_wind'] .
            (string)$character['character_earth'] .
            (string)$character['character_lightning'] .
            (string)$character['character_taijutsu'] .
            (string)$character['character_genjutsu'] .
            (string)$character['character_summon'] .
            (string)$character['character_control'] .
            (string)$character['character_bloodline'] .
            (string)$character['character_mission'] .
            (string)$character['character_ninja_essence'] .
            (string)$character['character_material'] .
            (string)$character['character_inv_hair'];

        $baseChar = $characterStr . $secret . $sessionKeyStr;
        $sha1Char = sha1($baseChar);

        $offsetChar = 0;
        if (strlen($sessionKeyStr) > 1) {
            $hexChar = substr($sessionKeyStr, 1, 1);
            if (ctype_xdigit($hexChar)) {
                $offsetChar = hexdec($hexChar);
            }
        }

        $character['character_hash'] = substr($sha1Char, $offsetChar, 12);
     
        $character['expiry_data'] = $expiryData;
        return $character;
    }
    
    /**
     * Create character
     * CharacterDAO.createCharacter(session_key, name, gender, hair_color, skin_color, hair, face)
     */
    public function createCharacter($sessionKey, $name, $gender, $hairColor, $skinColor, $hair, $face) {
        $account = validateSessionKey($sessionKey);
        if (!$account) {
            throw new Exception("Invalid session");
        }
        
        // Check if name already exists
        $stmt = $this->db->prepare("SELECT character_id FROM characters WHERE character_name = ?");
        $stmt->execute([$name]);
        if ($stmt->fetch()) {
            throw new Exception("Character name already exists");
        }
        
        // Create character
        $stmt = $this->db->prepare("
            INSERT INTO characters (
                account_id, character_name, character_gender, 
                character_hair_color, character_skin_color, 
                character_hair, character_face,
                character_level, character_xp,
                character_hp, character_max_hp,
                character_cp, character_max_cp,
                character_gold, character_skills, character_skill_talent, character_inventory
            ) VALUES (?, ?, ?, ?, ?, ?, ?, 1, 0, 100, 100, 100, 100, 300, '[]', '[]', '{}')
        ");
        $stmt->execute([
            $account['account_id'],
            $name,
            $gender,
            json_encode($hairColor),
            $skinColor,
            $hair . '_' . $gender,
            $face . '_' . $gender
        ]);
        
        $characterId = $this->db->lastInsertId();
        
        // Return created character
        return $this->getCharacterById($sessionKey, $characterId);
    }
    
    /**
     * Get extra data
     * CharacterDAO.getExtraData(session_key, hash, access_token)
     */
    public function getExtraData($sessionKey, $hash, $accessToken = null) {
        $account = validateSessionKey($sessionKey);
        if (!$account) {
            throw new Exception("Invalid session");
        }
        
        // Get main character
        $stmt = $this->db->prepare("
            SELECT character_id FROM characters 
            WHERE account_id = ? 
            ORDER BY character_id ASC 
            LIMIT 1
        ");
        $stmt->execute([$account['account_id']]);
        $character = $stmt->fetch();
        
        if (!$character) {
            return [];
        }
        
        // Get inventory items
        $stmt = $this->db->prepare("
            SELECT * FROM inventory_items 
            WHERE character_id = ?
        ");
        $stmt->execute([$character['character_id']]);
        $inventory = $stmt->fetchAll();
        
        // Get missions
        $stmt = $this->db->prepare("
            SELECT * FROM missions 
            WHERE character_id = ?
        ");
        $stmt->execute([$character['character_id']]);
        $missions = $stmt->fetchAll();
        
        // Get skills
        $stmt = $this->db->prepare("
            SELECT * FROM skills 
            WHERE character_id = ?
        ");
        $stmt->execute([$character['character_id']]);
        $skills = $stmt->fetchAll();
        
               // ------- Bangun extraData sesuai yang diharapkan parseCharacterData -------

        // PvP record default
        $pvpRecord = array(
            'play'                 => 0,
            'win'                  => 0,
            'lose'                 => 0,
            'disconnect'           => 0,
            'avg_level_diff'       => 0,
            'pvp_currency'         => 0,
            'pvp_point'            => 0,
            'pvp_tournament_ticket'=> 0,
        );

        // Data ekstra minimal aman
        $extraData = array(
            'pvp_record'          => (object)$pvpRecord,
            'canClaimFK'          => 0,
            'showNotice'          => 0,
            'noticeText'          => array(),   // harus Array, bukan string
            'claim_remain'        => 0,
            'showFanPage'         => 0,
            'event_170m_like'     => array(),   // harus Array, sama seperti claimTokenStatus
            'anni8thDoubleExpGold_times' => 0,
            'GodSnapCountDown'    => 0,
            'UnstoppableRageCountDown' => 0,
            'combine_boost_time'  => 0,
            'combine_boost_time_in_period' => 0,
            'combine_boost_time_period_start_left' => 0,
            'combine_boost_time_period_end_left'   => 0,
            'daily_login_data'    => array(),   // harus Array, bukan "" / null
            'file_1'              => array(),
            'file_2'              => array(),
            'file_3'              => array(),
            'get_learning_status' => array(),
            'training_skill'      => null,
            'sum_1'               => array(),
            'sum_2'               => array(),
            'sum_3'               => array(),

            // *** wajib array, dipakai tanpa null-check ***
            'player_pet'          => array(),
            'bloodline'           => array(),
            'senjutsu'            => array(),

            'senjutsu_system'     => array(),
            'invite_accepted'     => 0,
            'veteran_return_fk_accepted' => 0,
            'friendship_kunai'    => 0,
            'friend_accept_reward'=> array(),
            'total_friend_accepted'=> 0,
            'new_account_reward'  => 0,

            'consecutive_days'    => 0,
            'roulette_allowed'    => 0,
            'is_fan'              => 0,

            // nilai lain yang nanti dipakai di bawah
            'new_mail'            => 0,
            'newsfeed_material_posted'   => 0,
            'newsfeed_easter_2014_posted'=> 0,
            'character_create_date'      => '',
            'remaining_skill'            => 0,
            'reaming_pet'               => 0,
            'get_hunting_passport'      => 0,
            'DailyReward'               => null,
            'request_list_length'       => 0,
            'free_roulette_times'       => 0,
            'dailygift_gift_list'       => array(),
            'dailygift_request_limit'   => 0,
            'promotion'                 => array(),
            'promo_expired'             => 0,
            'daily_login'               => 0,
            'tutorial_expiry_item'      => 0,
            'option_data'               => array(
                'music_index'  => 0,
                'music_volume' => 1,
                'sound_on'     => 1,
            ),
            'lucky_spin_consecutive_day' => 0,
            'lucky_spin_remaining_spin'  => 0,
            'lucky_spin_show_wheel'      => 0,
            'lucky_spin_multiplier'      => array(),
            'is_hard_mode_locked'        => 0,
            'event_daily_login'          => null,
            'event_gift_bag'             => null,
            'new_clan_promote'           => array(
                'clan_status' => 0,
                'ad_page_max' => 0,
            ),
            'clan_id'                    => 0,
            'popup_arr'                  => array(),
            'layout'                     => array(),
                // Achievement & statistik karakter
    'achievement'           => array(),      // list ID achievement yang sudah didapat
    'achievement_point'     => 0,
    'recent_achievement'    => array(),      // list ID achievement terakhir
    'statistic_battle'      => new stdClass(), // atau array(), keduanya oke
    'char_statistic'        => new stdClass(), // dipakai Achievement.updateCharStat

            'char_login_per_day'         => 0,

            // ini placeholder, diisi di bawah
            'extra_data_hash'            => '',
        );

        // Tambahkan data inventory dari DB (opsional, tidak dipakai parseCharacterData)
        $extraData['inventory'] = $inventory;
        $extraData['missions']  = $missions;
        $extraData['skills']    = $skills;

        // ------- Hitung extra_data_hash seperti di parseCharacterData -------
        // pvpRecordStr dari pvp_record default 0
        $pvpRecordStr     = "0,0,0,0,0";
        $trainingSkillStr = "";
        $petsStr          = "";

        $extraDataString =
            $pvpRecordStr . "," .
            $trainingSkillStr . "," .
            $petsStr . "," .
            strval($extraData['consecutive_days']) . "," .
            ($extraData['roulette_allowed'] ? "1" : "0") . "," .
            ($extraData['is_fan'] ? "1" : "0");

        // Algoritma hash sama seperti ClientLibrary.getHash
        $sessionKeyStr = (string)$sessionKey;
        $secret        = 'Vmn34aAciYK00Hen26nT01';
        $base          = $extraDataString . $secret . $sessionKeyStr;
        $sha1          = sha1($base);

        $offset = 0;
        if (strlen($sessionKeyStr) > 1) {
            $hexChar = substr($sessionKeyStr, 1, 1);
            if (ctype_xdigit($hexChar)) {
                $offset = hexdec($hexChar);
            }
        }

        $extraData['extra_data_hash'] = substr($sha1, $offset, 12);

        return $extraData;
    }
}

